<?php
/* --------------------------------------------------------------
  ServiceFactory.inc.php 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

namespace StyleEdit\Factories;

use \StyleEdit\Repositories\ConfigRepository;
use \StyleEdit\Repositories\ConfigDeleter;
use \StyleEdit\Repositories\ConfigReader;
use \StyleEdit\Repositories\ConfigWriter;
use \StyleEdit\ConfigReadService;
use \StyleEdit\ConfigWriteService;
use \StyleEdit\ConfigSettings;

/**
 * Class ServiceFactory
 *
 * @package StyleEdit\Factories
 */
class ServiceFactory
{
	/** @var string $currentTemplate The name of the active template. */
	private $currentTemplate;
	
	/** @var \StyleEdit\Repositories\ConfigRepository $repository */
	private $repository;
	
	/** @var \StyleEdit\ConfigSettings $settings */
	private $settings;
	
	/** @var \StyleEdit\Factories\ConfigFactory $configFactory */
	private $configFactory;
	
	/** @var \StyleEdit\Repositories\ConfigReader $reader */
	private $reader;
	
	/** @var \StyleEdit\Repositories\ConfigWriter $writer */
	private $writer;
	
	/** @var \StyleEdit\Repositories\ConfigDeleter $deleter */
	private $deleter;
	
	
	/**
	 * Creates and Returns an Instance of ConfigReadService which is configured for the given template name.
	 *
	 * @param string $p_currentTemplate The name of the active template.
	 * @param bool   $useCache
	 *
	 * @throws \InvalidArgumentException if given template name is not a string or empty.
	 *
	 * @return \StyleEdit\ConfigReadService
	 */
	public function createReadService($p_currentTemplate, $useCache = true)
	{
		if(!is_string($p_currentTemplate) || empty($p_currentTemplate))
		{
			throw new \InvalidArgumentException('Invalid $p_currentTemplate argument value (string expected): '
			                                    . print_r($p_currentTemplate, true));
		}
		
		$this->currentTemplate = $p_currentTemplate;
		
		return new ConfigReadService($this->_getRepository($useCache));
	}
	
	
	/**
	 * Creates and Returns an Instance of ConfigWriteService which is configured for the given template name.
	 *
	 * @param string $p_currentTemplate The name of the active template.
	 *
	 * @throws \InvalidArgumentException if given template name is not a string or empty.
	 *
	 * @return \StyleEdit\ConfigWriteService
	 */
	public function createWriteService($p_currentTemplate)
	{
		if(!is_string($p_currentTemplate) || empty($p_currentTemplate))
		{
			throw new \InvalidArgumentException('Invalid $p_currentTemplate argument value (string expected): '
			                                    . print_r($p_currentTemplate, true));
		}
		
		$this->currentTemplate = $p_currentTemplate;
		
		return new ConfigWriteService($this->_getRepository(true), $this->_getConfigFactory());
	}
	
	
	/**
	 * Internal method to create and return a ConfigRepository object.
	 *
	 * @param bool $useCache
	 * 
	 * @return \StyleEdit\Repositories\ConfigRepository
	 */
	private function _getRepository($useCache)
	{
		if(!isset($this->repository) || !$useCache)
		{
			$this->repository = new ConfigRepository($this->_getReader($useCache), $this->_getWriter($useCache),
			                                         $this->_getDeleter($useCache));
		}
		
		return $this->repository;
	}
	
	
	/**
	 * Internal method to create and return a ConfigSettings object.
	 *
	 * @return \StyleEdit\ConfigSettings
	 */
	private function _getSettings()
	{
		if(!isset($this->settings))
		{
			$this->settings = new ConfigSettings($this->currentTemplate);
		}
		
		return $this->settings;
	}
	
	
	/**
	 * Internal method to create and return a ConfigFactory object.
	 *
	 * @return \StyleEdit\Factories\ConfigFactory
	 */
	private function _getConfigFactory()
	{
		if(!isset($this->configFactory))
		{
			$this->configFactory = new ConfigFactory();
		}
		
		return $this->configFactory;
	}
	
	
	/**
	 * Internal method to create and return a ConfigReader object.
	 *
	 * @param bool $useCache
	 *
	 * @return \StyleEdit\Repositories\ConfigReader
	 */
	private function _getReader($useCache)
	{
		if(!isset($this->reader) || !$useCache)
		{
			$this->reader = new ConfigReader($this->_getSettings(), $this->_getConfigFactory());
		}
		
		return $this->reader;
	}
	
	
	/**
	 * Internal method to create and return a ConfigWriter object.
	 *
	 * @param bool $useCache
	 *
	 * @return \StyleEdit\Repositories\ConfigWriter
	 */
	private function _getWriter($useCache)
	{
		if(!isset($this->writer) || !$useCache)
		{
			$this->writer = new ConfigWriter($this->_getSettings(), $this->_getConfigFactory());
		}
		
		return $this->writer;
	}
	
	
	/**
	 * Internal method to create and return a ConfigDeleter object.
	 *
	 * @param bool $useCache
	 *
	 * @return \StyleEdit\Repositories\ConfigDeleter
	 */
	private function _getDeleter($useCache)
	{
		if(!isset($this->deleter) || !$useCache)
		{
			$this->deleter = new ConfigDeleter($this->_getSettings());
		}
		
		return $this->deleter;
	}
}